(* the functions exposed to programs are at the bottom of the page
   printf_aux is a function that takes a format string and a list of arguments
   
   this function is not type safe, the arguments list has type ∀ α . (α list) but in reality the arguments can have different types 
   we use the "cast_*" family of functions to cast at compile time these elements of type "α" to elements of the correct type, which we know by looking at the format.

cast_to_float, cast_to_string, etc. are runtime primitives
some are just casts but others are conversions so should probably be renamed TODO *)

type integerformattype = Decimal | HexadecimalLower | HexadecimalUpper (* TODO: add binary *)

(* tables of characters *)
let decimal_chars = [| '0'; '1'; '2'; '3'; '4'; '5'; '6'; '7'; '8'; '9' |]
let hex_chars_lowercase = [| '0'; '1'; '2'; '3'; '4'; '5'; '6'; '7'; '8'; '9'; 'a'; 'b'; 'c'; 'd'; 'e'; 'f' |]
let hex_chars_uppercase = [| '0'; '1'; '2'; '3'; '4'; '5'; '6'; '7'; '8'; '9'; 'A'; 'B'; 'C'; 'D'; 'E'; 'F' |]


let format_integer i format_type = 
  let chars_table, base = 
    case format_type  
     | Decimal -> decimal_chars, 10L
     | HexadecimalLower -> hex_chars_lowercase, 16L
     | HexadecimalUpper -> hex_chars_uppercase, 16L
    end in 

  let rec format_aux i accu = 
    if Int64.compare i base < 0  
    then chars_table[Int64.to_int i] :: accu
    else format_aux  (Int64.div i base)  (chars_table[Int64.to_int (Int64.rem i base)] :: accu)   in 

  let chars = format_aux (if Int64.compare i 0L >= 0 then i else Int64.mul (Int64.lognot 0L) i) [] in (* we only pass a positive integer *)
  let chars_with_sign = if Int64.compare i 0L >= 0 || format_type <> Decimal then chars else '-' :: chars in (* add a minus for negative numbers in decimal formatting *)
  let chars_with_prefix = if format_type = HexadecimalLower || format_type = HexadecimalUpper then '0' :: 'x' :: chars_with_sign else chars_with_sign in (* "0x" prefix for hexadecimal formatting *)
  let s = String.build (List.length chars_with_prefix) in 
  let () = List.iteri (fun i c -> String.set s i c) chars_with_prefix in 
  s
    

let format_float f = 
  let integral_part = int_of_float f in 
  let decimal_part = Int64.of_int (int_of_float ((f -. float_of_int integral_part) *. 10E6)) in (* TODO: what is this 10E6? document the size of the fractional part *)
  (format_integer  (Int64.of_int integral_part)  Decimal) ^ "." ^ (format_integer decimal_part Decimal)
  

(* TODO: handle escaping and more complex formats, use the same code as in type inference 
   we have to create new strings, using blit would be too complicated *)
let rec printf_aux  format_string  pos  arguments  accu = 
  match arguments with 
      [] -> let s = String.sub format_string pos (String.length format_string - pos) in 
            accu ^ s
    | arg :: rest_of_arguments -> 
        let p_opt = String.indexoffrom "%" format_string pos in (* TODO: take double escaping into account, TODO: assume the string has already been validated *)
        match p_opt with 
            Some p -> 
            if p < String.length format_string - 1 then (* TODO: we could assume that the string has been validated by the type inference pass *)
              let result = 
                case format_string.[p + 1]  
                  | 'f' -> (cast_to_float arg) ▷ format_float (* careful when using the "cast_*" functions *)
                  | 's' -> cast_to_string arg
                  | 'i' -> let i = cast_to_int arg in format_integer (int64_of_int i) Decimal
                  | 'd' -> let i = cast_to_int arg in format_integer (int64_of_int i) Decimal 
                  | 'x' -> let i = cast_to_int arg in format_integer (int64_of_int i) HexadecimalLower
                  | 'X' -> let i = cast_to_int arg in format_integer (int64_of_int i) HexadecimalUpper
                  | 'c' -> let s = String.build 1 in let () = s.[0] <- cast_to_char arg in s
                  | 'b' -> let b = cast_to_int arg in if b = 1 then "true" else "false"
                  | 'L' -> 
                    let i = cast_to_int64 arg in 
                    case format_string.[p + 2]
                     | 'i' -> format_integer i Decimal
                     | 'x' -> format_integer i HexadecimalLower
                     | 'X' -> format_integer i HexadecimalUpper
                     | _ -> let () = print_string "unsupported format string in printf module" in ""
                    end
                  | '0' -> 
                    let padding_to = int_of_char format_string.[p + 2] - 48 in (* TODO: do double-digits paddings? *)
                    let s = 
                      case format_string.[p + 3]
                       | 'i' -> let i = cast_to_int arg in format_integer (int64_of_int i) Decimal
                       | 'x' -> let i = cast_to_int arg in format_integer (int64_of_int i) HexadecimalLower
                       | 'X' -> let i = cast_to_int arg in format_integer (int64_of_int i) HexadecimalUpper
                       | _ -> let () = print_string "unsupported format string in printf module" in ""
                      end in 
                    if String.length s >= padding_to then 
                      s
                    else
                      let padding = String.build (padding_to - String.length s) in 
                      let _ = String.initialize padding '0' in  
                      padding ^ s
                  | _ -> let () = print_string "unsupported format string in printf module" in ""
                end in 
              let s = String.sub format_string pos (p - pos) in 
              let newpos = case format_string.[p + 1] | 'L' -> p + 3 | '0' -> p + 4 | _ -> p + 2 end in 
              printf_aux  format_string  newpos  rest_of_arguments  (accu ^ s ^ result)
            else
              accu
          | None -> accu
        end
  end
  

(* functions exposed to programs *)
let printf_list  format_string  arguments = 
  let s = printf_aux  format_string  0  arguments  ""  in 
  print_string s
  
let sprintf_list  format_string  arguments = 
  printf_aux  format_string  0  arguments  ""

let print_strings_list l = 
  let () = print_string "[ " in
  let rec print_loop l = 
    match l with 
        [] -> ()
      | t :: q -> let () = print_string t in let () = print_string "; " in print_loop q
    end in 
  let () = print_loop l in 
  print_string "]"
